# Code for Project NETC 15-1 Use of Forested Habitat Adjacent to Highways by Northern Long-Eared Bats
#By:
#Dr. Jeffrey Foster, PI
#Dr. Dan Linden, Co-PI
#Dr. Erik Blomberg, Co-PI
#Dr. Marina Fisher-Phelps, Postdoctoral Research Associate
#Katherine Ineson, Graduate Student Assistant

# Commented out code was not used in the final model due to overall code or model changes or preliminary analysis excluded the step
#setwd("~/GitHub/NETC_15-1-NLEB-Models") ----Please set to where data is stored on your own computer
library(jagsUI)
library(tidyr)
library(dplyr)
library(ggplot2)
library(reshape2)
library(raster)
library(proxy)
library(viridis)

dat <- read.csv("AllData_JAGS_11092018.csv")

# KnotDis<-read.csv("GeoDis_KnotstoKnots50km.csv")
# KnotLocalDis<-read.csv("GeoDis_LocalstoKnots50km.csv")
# 
# KnotDis.mat <- matrix(NA,nrow=max(KnotDis$INPUT_FID),ncol=max(KnotDis$NEAR_FID))
# for(i in 1:nrow(KnotDis.mat)){
#   for(j in 1:ncol(KnotDis.mat)){
#     if(i==j){
#       KnotDis.mat[i,j] <- 0
#     } else {
#       KnotDis.mat[i,j] <- KnotDis$DsStarStar[which(KnotDis$INPUT_FID==i & KnotDis$NEAR_FID==j)]
#     }
#     }
#   }
# 
# KnotLocalDis.mat <- matrix(NA,nrow=max(KnotLocalDis$NEAR_FID),ncol=max(KnotLocalDis$INPUT_FID))
# for(i in 1:nrow(KnotLocalDis.mat)){
#   for(j in 1:ncol(KnotLocalDis.mat)){
#     KnotLocalDis.mat[i,j] <- KnotLocalDis$DsStar[which(KnotLocalDis$NEAR_FID==i & KnotLocalDis$INPUT_FID==j)]
#     }
#   }

knots <- shapefile("./Shapefiles/NEbufferKnots_1029")
knots.xy <- knots@coords; names(knots.xy) <- c("x","y")

# knot to site distances (in meters)
KnotLocalDis.mat <- proxy::dist(x=data.frame(x=dat$POINT_X,y=dat$POINT_Y), 
                                y=knots.xy, method = "euclidean")
# knot to knot distances (in meters)
KnotDis.mat <- proxy::dist(x=knots.xy, y=knots.xy, method = "euclidean")


dat$LanesMinus<-dat$NumberLanes-2
dat$WidthScale<-scale(dat$LaneWidth)
dat$ADDTscale<-scale(dat$aadt)
dat$SpeedLimScale<-scale(dat$SpeedLimit)
dat$SurveyScale<-log(dat$SurveyNumber)
# dat$GrassScrubScale<-scale(dat$GrassScrub)
# dat$DevScale<-scale(dat$Developed)
dat$ForestScale<-scale(dat$TotalForest)
dat$WaterDisTrans <- sqrt(dat$WaterDistance/1000)
#dat$RoadDensTrans <- sqrt(dat$RoadDensity_Meters/100000)
dat$RoadDensNLTrans <- sqrt(dat$NoLocal_RoadDen/100000)
# dat$AGScale<-scale(dat$Agriculture)
# dat$ScaleDec<-scale(dat$DeciduousForest)
# dat$ScaleEG<-scale(dat$EverGForest)
# dat$WaterScale<-scale(dat$Water)
# dat$WetlandScale<-scale(dat$Wetland)
# dat$TreeH1Scale<-scale(dat$TreeHeight_0_5)
# dat$TreeH2Scale<-scale(dat$TreeHeight_5_10)
# dat$TreeH3Scale<-scale(dat$TreeHeight_10_25)
# dat$TreeH4Scale<-scale(dat$TreeHeight_25_50)
# dat$TreeCoverScale<-scale(dat$TreeCover)

#load raster files below
forest.r <- raster("./Rasters/Updated/Forest.tif")
roads.r <- raster("./Rasters/Updated/RoadsNoLocal.tif")
water.r <- raster("./Rasters/Updated/Water.tif")

Forest.extract<-extract(forest.r,dat[,c("POINT_X","POINT_Y")])
Road.extract<-extract(roads.r,dat[, c("POINT_X","POINT_Y")])
Water.extract<-extract(water.r,dat[, c("POINT_X","POINT_Y")])
dat$ForestEx<-scale(Forest.extract)
dat$RoadEx<-sqrt(Road.extract/100000)
dat$WaterEx<-sqrt(Water.extract/1000)

newdat <- dat %>%
  gather(v, value, Date_1:Night_14) %>%
  separate(v, c("var","survey"),sep="_") %>%
  spread(var, value) %>%
  filter(!is.na(Night))

newdat <- newdat[order(newdat$MFP_Site,newdat$Date),]

#missing Dates need to be fixed
newdat$Date[is.na(newdat$Date)] <- median(newdat$Date,na.rm=T)
newdat$DateScale<-scale(newdat$Date)

# Bundle Data
win.data<-list(y=newdat$Night,
               
               #-- site occupancy variables
               #---------------------------
               #Forest=newdat$ForestScale,
               #WaterDis=newdat$WaterDisTrans,
               #RoadDen=newdat$RoadDensTrans,
               #RoadDenNL=newdat$RoadDensNLTrans,
               ForestEx=newdat$ForestEx,
               RoadEx=newdat$RoadEx,
               WaterEx=newdat$WaterEx,
               # wetland=newdat$WetlandScale, 
               # AG=newdat$AGScale, 
               # water=newdat$WaterScale,
               # TreeH1=newdat$TreeH1Scale, 
               # TreeH2=newdat$TreeH2Scale,
               # TreeH3=newdat$TreeH3Scale,
               # TreeH4=newdat$TreeH4Scale,
               # TreeCover=newdat$TreeCoverScale,
               # AADT=newdat$ADDTscale,
               # GrassScrub=newdat$GrassScrubScale,
               # Develop=newdat$DevScale,
               # DecForest=newdat$ScaleDec, 
               # EGforest=newdat$ScaleEG,
               # RoadWidth=newdat$WidthScale,
               # speed=newdat$SpeedLimScale, 
               # speedlimit=newdat$SpeedLimScale,
               
               #-- detection variables
               #---------------------------
               Lanes=newdat$LanesMinus, 
               SurveyNum=newdat$SurveyScale, 
               Date=newdat$DateScale, 
               Year=newdat$Year, 

               #-- knot distances (km)
               #---------------------------
               d.s.star=KnotLocalDis.mat/1000,
               d.s.star.star=KnotDis.mat/1000,
               N.star=nrow(KnotDis.mat),
               
               #-- survey design
               #---------------------------
               M=length(unique(newdat$MFP_Site)), 
               J=nrow(newdat), 
               site=match(newdat$MFP_Site,unique(newdat$MFP_Site)))


# need to grab a single site covariate from the vertical data
win.data$sitez <- sapply(sort(unique(win.data$site)),function(x){min(which(win.data$site==x))})

# prior scale (intercept and coefficients)
win.data$tau.b <- c(0.30,0.04,0.04,0.04)


cat("
    model {
    
    # Priors
    mean.p ~ dunif(0, 1)          # Detection intercept on prob. scale
    alpha0 <- logit(mean.p)       #   same on logit scale
    #mean.psi ~ dunif(0, 1)        # Occupancy intercept on prob. scale
    #beta0 <- logit(mean.psi)      #   same on logit scale

    sigmasq <- 1/sigmasq.inv    # Spatial process priors
    sigmasq.inv ~ dgamma(2,1)      # Spatial process priors
    phi ~ dgamma(1,0.1)            # Spatial process priors


    tausq = 1/tausq.inv           # Spatial process priors
    tausq.inv ~ dgamma(0.1,0.1)   # Spatial process priors
    
    for(k in 1:4){                # Covariate on logit(detection)
    alpha[k] ~ dnorm(0, 0.04)
    }      
    for(b in 1:4){                # Terms in occupancy model (intercept + 3 coefficients)
    beta[b] ~ dnorm(0, tau.b[b])
    }
    
    # Likelihood
    for (i in 1:M) {              # Loop over sites
    z[i] ~ dbern(psi[i])
    logit(psi[i]) <- w.tilde[i] +      # spatially varying intercept
      # spatially varying coefficients
      beta[2]*RoadEx[sitez[i]] +
      beta[3]*ForestEx[sitez[i]] +
      beta[4]*WaterEx[sitez[i]] +
      # nonspatial residual error
      e.i[i]
    
    # Residual error (w/ corrections)
    e.i[i] ~ dnorm(0, prec[i])
    prec[i] <- 1/var.all[i]
    var.all[i] <- tausq + 
      sigmasq[1] - correction[i] 

    } #end i

    
    for (j in 1:J) {              # Loop over all observations
    y[j] ~ dbern(z[site[j]] * p[j])
    logit(p[j]) <- alpha0 +                # detection (p) intercept
    alpha[1] * ForestEx[j] +
    alpha[2] * Date[j] +               
    alpha[3] * Date[j]^2 +                      
    alpha[4] * Year[j]
    } #end j
 
    
# Spatial Predictive process

    w.tilde.star[1:N.star] ~ dmnorm(mu.w.star[1:N.star], C.star.inv[1:N.star,1:N.star])
    C.star.inv[1:N.star,1:N.star] = inverse(C.star[1:N.star,1:N.star])
    
    for (i in 1:N.star) {
      mu.w.star[i] = beta[1]
      C.star[i,i] = sigmasq
      for (j in 1:(i-1)) {
        C.star[i,j] = sigmasq*exp(-(d.s.star.star[i,j]/phi))
        C.star[j,i] = C.star[i,j]
    } }
# Interpolate spatial PP back on to original sites
    for(i in 1:M) {
      for(j in 1:N.star) {
        C.s.star[i,j] = sigmasq*exp(-(d.s.star[i,j]/phi))
      } }
    w.tilde[1:M] = C.s.star[1:M,1:N.star]%*%C.star.inv[1:N.star,1:N.star]%*%w.tilde.star[1:N.star]
# Variance correction
    for(i in 1:M){
      correction[i] = t(C.s.star[i,1:N.star])%*%C.star.inv[1:N.star,1:N.star]%*%C.s.star[i,1:N.star]
    }
    



    }
    ",file = "modelKnotNoLocalEX.txt")


# Inits
inits <- function(){list(z = sapply(1:win.data$M,function(x){max(win.data$y[win.data$site==x])}), 
                         beta = c(-1.5,0,0,0), mean.p = runif(1))}

# Parameters monitored
params <- c("alpha0", "alpha", "beta",
            "phi","sigmasq","tausq","w.tilde","w.tilde.star")

# MCMC settings
ni <- 50000   ;   nt <- 5   ;   na <- 50000   ;   nc <- 3
#ni <- 400   ;   nt <- 1   ;   na <- 200   ;   nc <- 3


# Run JAGS and summarize

(start <- Sys.time())
fit <-jags(win.data, inits, params, "modelKnotNoLocalEX.txt", 
           n.chains = nc, n.thin=nt, n.iter = ni, n.adapt = na, 
           parallel = ifelse(nc>1,TRUE,FALSE))
(end <- Sys.time()-start)

print(fit, 3)
#plot(fit)

save(fit, file=paste0("JAGS_1km_",Sys.Date(),".RData"))

# load saved model object
load(file.choose())

# predicted occupancy
plot(dat$Longitude,dat$Latitude,asp=1,
     cex=plogis(fit$q50$w.tilde[1,] + 
                  fit$q50$w.tilde[2,]*dat$RoadEx + 
                  fit$q50$w.tilde[3,]*dat$ForestEx +
                  fit$q50$w.tilde[4,]*dat$WaterEx
     )*10)

# road density coefficients
plot(dat$Longitude,dat$Latitude,asp=1,
     cex=fit$q50$w.tilde+4)


plot(dat$Longitude,dat$Latitude,asp=1,pch=16,
     col=viridis(22)[round(plogis(
       fit$q50$w.tilde[1,] + 
         fit$q50$w.tilde[2,]*dat$RoadEx + 
         fit$q50$w.tilde[3,]*dat$ForestEx +
         fit$q50$w.tilde[4,]*dat$WaterEx
       )*100,0)+1])
points(dat$Longitude,dat$Latitude,cex=dat$MYSE_PRES,pch=3)



# tildes <- apply(fit$sims.list$w.tilde,c(2:3),
#                  function(x){quantile(x,probs=c(0.025,0.5,0.975))})
tildes <- apply(fit$sims.list$w.tilde.star,2,
                function(x){quantile(x,probs=c(0.025,0.5,0.975))})

dimnames(tildes) <- list(quantile=c("lower95","median","upper95"),
                         #betas=c("Intercept","RoadEx","ForestEx","WaterEx"),
                         sites=1:nrow(knots)
                         #sites=1:win.data$M)
)

tildes.lng <- melt(tildes)
tildes.wide <- spread(tildes.lng,quantile,value)
tildes.wide$sites <- as.factor(tildes.wide$sites)

#ordering the site-specific coefficients

tildes.wide %>% 
  #filter(betas=='Intercept') %>%
  ggplot(aes(x=reorder(sites,median), y=plogis(median))) +
  geom_point(pch=15,size=1.5) +
  #geom_hline(yintercept=0,col="red",lty=2,lwd=1.25)+
  theme_minimal()+
  theme(axis.text.x=element_blank(),
        axis.ticks.x=element_blank()) +
  labs(x="Spatial knot",y="Occupancy probability",
       title="Variation in average occupancy probability at knots") +
  geom_errorbar(aes(ymin=plogis(lower95),ymax=plogis(upper95)),col=rgb(0,0,0,.3),lwd=.75)
ggsave(filename="Intercept_tildes.png",dpi=300,width=7.5,height=5)  


#plotting the knots


# making landscape predictions

# stack of all 3 layers
land.r <- stack(forest.r,roads.r,water.r)
# extract values and coords
land.val <- na.omit(values(land.r))
land.df <- as.data.frame(land.val)
land.xy <- coordinates(land.r)[-na.action(land.val),]

# knot to cell distances
dist_cell_knot <- proxy::dist(x=land.xy, y=knots.xy, method = "euclidean")/1000
dist_knot_knot <- proxy::dist(x=knots.xy, y=knots.xy, method = "euclidean")/1000



# ForestEx already scaled in dat -- just need the attributes
land.df$ForestZ <- (land.df$Forest-attr(dat$ForestEx,"scaled:center"))/attr(dat$ForestEx,"scaled:scale")

land.df$WaterZ <- sqrt(land.df$Water/1e5)
land.df$RoadsNoLocalZ <- sqrt(land.df$RoadsNoLocal/1e5)

w.tilde.star <- apply(fit$sims.list$w.tilde.star,2,median)
sigmasq <- fit$q50$sigmasq
phi <- fit$q50$phi

# matrices of covariance
C.star <- sigmasq*exp(-(dist_knot_knot/phi))
C.star.inv <- solve(C.star)
C.s.star <- sigmasq*exp(-(dist_cell_knot/phi))

cell.tilde.star <- C.s.star %*% C.star.inv %*% w.tilde.star
#}

cell.logit.psi <- (cell.tilde.star
                   + fit$q50$beta[4] * land.df$WaterZ
                   
                   )

# raster of cell predictions
land.pred <- rasterFromXYZ(data.frame(land.xy,pred=plogis(cell.logit.psi)))

png(file="./Outputs/predicted_psi.png",height=8,width=6,res=600,units="in")
plot(land.pred,col=rev(viridis(12)))
points(dat[,c("POINT_X","POINT_Y")],pch=3,col="gray50")
points(dat[,c("POINT_X","POINT_Y")],pch=16,cex=dat$MYSE_PRES)
dev.off()



